//=============================================================================
// SimpleVoiceControl.js
//=============================================================================

/*:
 * @plugindesc SimpleVoiceControl V1.0.1
 * @author NJ
 * 
 * @param VoiceCutSwitch
 * @type switch
 * @desc ボイスカットを制御するスイッチID。このスイッチがONの場合、メッセージが進むと再生中のボイスを停止します。
 * @default 1
 * 
 * @param EnableVoicePlayback
 * @type boolean
 * @text ボイス再生の有効化
 * @desc ボイス再生を有効にするかどうかを設定します。[true: 再生する / false: 再生しない]
 * @default true
 * 
 * @help
 * このプラグインはSimpleVoiceプラグインを拡張し、以下の機能を提供します。
 * 
 * - 制御文字`\SV[ファイル名]`でボイスを再生
 * - 指定スイッチがONの場合、メッセージ終了時に再生中のボイスを停止
 * - 再生機能をプラグインパラメータで有効/無効を切り替え可能
 *
 * 使用例:
 * \SV[voice1] => ボイス「voice1」を再生
 * 
 * 注意:
 * - SimpleVoiceプラグインが必要です。
 * - プラグイン管理でSimpleVoiceの「下」に配置してください。
 *
 * バージョン
 * V1.0.0 初回
 * V1.0.1 ボイス再生の有効化機能を追加。
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(function() {
    'use strict';

    // プラグインパラメータを取得
    const parameters = PluginManager.parameters('SimpleVoiceControl');
    const voiceCutSwitchId = Number(parameters['VoiceCutSwitch'] || 1);
    const enableVoicePlayback = parameters['EnableVoicePlayback'] === 'true';

    let lastPlayedVoiceName = null; // 最後に再生したボイス名を記録

    // 制御文字を処理する関数を拡張
    const _Window_Message_processEscapeCharacter = Window_Message.prototype.processEscapeCharacter;
    Window_Message.prototype.processEscapeCharacter = function(code, textState) {
        if (code === 'SV') {
            const fileName = this.obtainEscapeParam(textState);
            if (fileName) {
                if (enableVoicePlayback) { // ボイス再生が有効な場合のみ再生を行う
                    lastPlayedVoiceName = fileName; // 再生中のボイス名を記録
                    AudioManager.playVoice({
                        name: fileName,
                        volume: ConfigManager.voiceVolume || 100,
                        pitch: 100,
                        pan: 0
                    });
                }
            }
        } else {
            _Window_Message_processEscapeCharacter.call(this, code, textState);
        }
    };

    // メッセージ終了時にボイスを停止
    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        if ($gameSwitches.value(voiceCutSwitchId) && lastPlayedVoiceName) {
            AudioManager.stopVoice(lastPlayedVoiceName); // 再生中のボイスを停止
            lastPlayedVoiceName = null;
        }
        _Window_Message_terminateMessage.call(this);
    };

    // 制御文字のパラメータ取得処理
    const _Window_Message_obtainEscapeParam = Window_Message.prototype.obtainEscapeParam;
    Window_Message.prototype.obtainEscapeParam = function(textState) {
        const regExp = /^\[([^\]]+)\]/;
        const arr = regExp.exec(textState.text.slice(textState.index));
        if (arr) {
            textState.index += arr[0].length;
            return arr[1];
        }
        return _Window_Message_obtainEscapeParam.call(this, textState);
    };

})();
